% Copyright (c) 2013, Massachusetts Institute of Technology
% This program was presented in the book "Visual Psychophysics:
% From Laboratory to Theory" by Zhong-Lin Lu and Barbara Dosher.
% The book is available at http://mitpress.mit.edu/books/visual-psychophysics

%%% Program RatingExperiment.m 

function RatingExperiment 

%% Display Setup Module

% Define display parameters

whichScreen = max(Screen('screens'));
p.ScreenDistance = 30; 	% in inches
p.ScreenHeight = 15; 	% in inches
p.ScreenGamma = 2;	% from monitor calibration
p.maxLuminance = 100; % from monitor calibration
p.ScreenBackground = 0.5;

% Open the display window and hide the mouse cursor

if exist('onCleanup', 'class'), oC_Obj = onCleanup(@()sca); end % close any pre-existing PTB Screen window
%Prepare setup of imaging pipeline for onscreen window. 
PsychImaging('PrepareConfiguration'); % First step in starting pipeline
PsychImaging('AddTask', 'General', 'FloatingPoint32BitIfPossible');   % set up a 32-bit floatingpoint framebuffer
PsychImaging('AddTask', 'General', 'NormalizedHighresColorRange'); % normalize the color range ([0, 1] corresponds to [min, max])
PsychImaging('AddTask', 'General', 'EnablePseudoGrayOutput'); % enable high gray level resolution output with bitstealing
PsychImaging('AddTask', 'FinalFormatting', 'DisplayColorCorrection', 'SimpleGamma');  % setup Gamma correction method using simple power function for all color channels 
[windowPtr p.ScreenRect] = PsychImaging('OpenWindow', whichScreen, p.ScreenBackground);  % Finishes the setup phase for imaging pipeline, creates an onscreen window, performs all remaining configuration steps
PsychColorCorrection('SetEncodingGamma', windowPtr, 1 / p.ScreenGamma);  % set Gamma for all color channels
HideCursor;  % Hide the mouse cursor

% Get frame rate and set screen font

p.ScreenFrameRate = FrameRate(windowPtr);
Screen('TextFont', windowPtr, 'Times'); 
Screen('TextSize', windowPtr, 24);

%% Experimental Module

% Specify the stimulus
p.stimSize = 6;     % image diameter size in visual degree
p.contrast = 0.01;  % Gabor contrast
p.sf = 0.9; % c/d
p.sigma = 1.1; % degree
p.stimDuration = 0.1;  % stim duration in seconds
p.fixDuration = 0.15;  % fixation duration in seconds
p.ITI = 0.5;           % seconds between trials
nTrials = 300;         % total number of trials
keys = {'1' '2' '3' '4' '5' '6' 'esc'}; % response keys
 
% Compute stimulus parameters
ppd = pi / 180 * p.ScreenDistance / p.ScreenHeight * ...
      p.ScreenRect(4);          % pixels per degree
m = round(p.stimSize * ppd);    % stimulus size in pixels
sf = p.sf / ppd;                % cycles per pixel
sc = round(p.sigma * ppd);      % sigma in pixels
params = [180 sf sc 0 1 0 0 0]; % gabor parameters: phase sf
                                % sigma contrast
fixLen = 32;           % length of fixation in pixels
[xc yc] = RectCenter(p.ScreenRect);
fixXY = [[-1 0 0 1]*fixLen + [-1 -1 1 1 ]*m/2 + xc ...
        [1 1 1 1]*xc; [1 1 1 1]*yc [-1 0 0 1]*fixLen + ...
        [-1 -1 1 1 ] * m / 2 + yc]; 
       
p.randSeed = ClockRandSeed;    % use clock to set seed for
                               % the random number generator
 
% procedural gabor allows us to change its parameters 
% very quickly 
tex = CreateProceduralGabor(windowPtr, m, m, 0, ...
      [1 1 1 0] * 0.5, 1, 0.5);

% Initialize a table to set up experimental conditions
p.recLabel = {'trialIndex' 'haveGabor' 'rating' 'respTime'};
rec = nan(nTrials, length(p.recLabel)); 
        % matrix rec is made of nTrials x 4 of NaN
rec(:, 1) = 1 : nTrials;    
        % count the trial numbers from 1 to nTrials
haveGabor = repmat([0 1], 1, ceil(nTrials / 2));
rec(:, 2) = Shuffle(haveGabor(1 : nTrials)); 
        % shuffle 0s and 1s
 
% Prioritize display to optimize display timing
Priority(MaxPriority(windowPtr));

% Start experiment with instructions
str = ['Press 1, 2, 3 as high, middle, and low confidence '...
     'Gabor absent responses\n\n' 'Press 4, 5, 6 as low, ' ...
     'middle and high confidence Gabor present responses ' ...
     '\n\n' 'Press SPACE to start.'];
DrawFormattedText(windowPtr, str, 'center', 'center', 1);
        % Draw Instruction text string centered in window
Screen('Flip', windowPtr);  
        % flip the text image into active buffer
WaitTill('space');      % wait till space bar is pressed
Secs = Screen('Flip', windowPtr);   % turn off instruction
p.start = datestr(now); % record start time
 
% Run nTrials trials
for i = 1 : nTrials
    Screen('DrawLines', windowPtr, fixXY, 3, 0.3);  
         % fixation crosshairs        
    t0 = Screen('Flip', windowPtr, Secs + p.ITI, 1);
    
    if rec(i, 2), params(4) = p.contrast;
    else params(4) = 0;
    end
    Screen('DrawTexture', windowPtr, tex, [], [], 0, [], ...
           [], [], [], 2, params);
    t0 = Screen('Flip', windowPtr, t0 + p.fixDuration);
    Screen('Flip', windowPtr, t0 + p.stimDuration);
       
    [key Secs] = WaitTill(keys);      % wait till response
    if iscellstr(key), key = key{1}; end 
         % take the first in case of multiple keys
    if strcmp(key, 'esc'), break; end % to stop
    rec(i, 3 : 4) = [str2double(key)  Secs-t0]; 
         % record rating and respTime
end
 
p.finish = datestr(now);            % record finish time
save RatingExperiment_rst rec p;    % save the results

% Tabulate the result
nPresent = zeros(1, 6);
nAbsent  = zeros(1, 6);
rec(i : end, :) = [];    % remove unfinished trials in 
                         % case of early exit
for i = 1 : 6
    nPresent(i) = sum(rec(rec(:, 2) == 1, 3) == i);
    nAbsent(i)  = sum(rec(rec(:, 2) == 0, 3) == i);
end
fmtStr = repmat('%5.0f', 1, 6);
fprintf('\n');
fprintf(['Rating ' fmtStr '   total\n'], 1 : 6);
fmtStr = repmat('%5.0f', 1, 7);
fprintf(['Present' fmtStr '\n'], nPresent, sum(nPresent));
fprintf(['Absent ' fmtStr '\n'], nAbsent,  sum(nAbsent));
